<?php
namespace Upload;

use CodeIgniter\Controller;
use CodeIgniter\HTTP\CLIRequest;
use CodeIgniter\HTTP\IncomingRequest;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\HTTP\UserAgent;
use CodeIgniter\Files\File;

use Psr\Log\LoggerInterface;

use CodeIgniter\I18n\Time;
use CodeIgniter\I18n\TimeDifference;

use Store\DataHelper;
use Constant\ConstantHelper;

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class UploadHelper extends Controller {
	public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger) {
		parent::initController($request, $response, $logger);
	}

	/***************************************************************
	*
	*			Single image					
	*
	***************************************************************/

	/**
	* Faz o upload de uma única imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param string  $modulo 
	* @param string  $atual 
	* @param string  $pasta 
	* @param string  $subpasta 
	* @param string  $index 
	* @param int 	 $id 
	* @param int 	 $keyArray 
	* @param boolean $saveDB 
	* 
 	* @return strint | array
	*/

 	function UPH_uploadImage($modulo, $atual, $pasta, $subpasta, $index, $id, $keyArray = 0, $saveDB = true, $webp = false) {
 		$this->dataHelper = new DataHelper;
 		$db = \Config\Database::connect();

 		if(isset($_FILES[$index])) {

 			/*
 			* Mensagem de retorno
 			*/

 			$return = array(
 				'status' => 0,
 				'msg'		 => '',
 				'img'		 => ''
 			);

			/*
			* Valida a pasta
			*/

			$path = PUBLICUPD . $pasta;

			if(!is_dir($path)) {
				mkdir($path);
			}

			if(isset($subpasta) AND !empty($subpasta)) {
				$path = PUBLICUPD . $pasta . '/' . $subpasta;

				if(!is_dir($path)) {
					mkdir($path);
				}
			}

			/*
			* Pega o arquivo enviado
			*/

			if(!empty($keyArray)) {
				$imagem = service('request')->getFile($index . '.' . $keyArray);
			}
			else {
				$imagem = service('request')->getFile($index);
			}

			/*
			* Valida o tipo de arquivo enviado
			*/

			$mimeTypes = array(
				'image/jpg',
				'image/jpeg',
				'image/gif',
				'image/png',
				'image/webp',
			);

			$mimeActual = $imagem->getClientMimeType();

			if(!in_array($mimeActual, $mimeTypes)) {
				$return['status']  = 0;
				$return['msg'] 		.= "O tipo de imagem enviada é inválido. <br/>";
			}

			/*
			* Pega a extensão do arquivo
			*/

			$file 			= new File($imagem);
			$extension 	= $file->guessExtension();

			/*
			* Valida o tamanho do arquivo enviado
			*/

			$config = $this->dataHelper->DH_getAjustes();

			if($imagem->getSizeByUnit('kb') > $config->upload_maxsize) {
				$return['status']  = 0;
				$return['msg'] 		.= "O tamanho do arquivo é inválido, tamanho máximo permitido: " . $config->upload_maxsize . " KB <br/>";
			}

			if($imagem->isValid() AND !$imagem->hasMoved()) {
				$name = $imagem->getRandomName();
				$imagem->move($path, $name);

				if(isset($atual) && $atual != "") {
					$this->UPH_deleteImage($modulo, $index, $pasta, $subpasta, $atual, $id, $saveDB);
				}

				if ($webp) {
					if (file_exists($path . '/' . $name)) {
						$fileActual = $path . '/' . $name;

						$file      = new File($fileActual);
						$extension = $file->guessExtension();

						$fileWebp = $path . '/' . str_replace($extension, 'webp', $name);
						service('image')->withFile($fileActual)->convert(IMAGETYPE_WEBP)->save($fileWebp);

						$name = str_replace($extension, 'webp', $name);
					}
				}
				else {
					if(file_exists($path . '/' . $name)) {
						$fileActual = $path . '/' . $name;

						$file 			= new File($fileActual);
						$extension 	= $file->guessExtension();
					}
				}

				if($saveDB == true) {
					$table 		= $modulo;
					$builder  = $db->table($table);

					$builder->set($index, $name)->where('id', $id);
					$builder->update();
				}

				$mod = $this->dataHelper->DH_getModulo($modulo);

				if(isset($atual) AND !empty($atual)) {
					$return['status'] = 1;
					$return['msg'] 		= "trocou o arquivo <b>" . $atual . "</b> do módulo <b>"  . $mod->titulo . "</b> pelo <b>" . $name . '</b>. ID do cadastro: <b>' . $id . "</b>";
					$return['img'] 		= $name;
				}
				else {
					$return['status'] = 1;
					$return['msg'] 		= "salvou o arquivo <b>" . $name . "</b> no módulo <b>"  . $mod->titulo . '</b>. ID do cadastro: <b>' . $id . "</b>";
					$return['img'] 		= $name;
				}
			}
			else {
				$return['status']  = 0;
				$return['msg'] 		.= "Erro ao tentar fazer o envio.";
			}
		}
		else {
			$return['status']  = 0;
			$return['msg'] 		 = "Arquivo não encontrado.";
		}

		return $return;
	}

	/**
	* Deleta uma única imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param string  $modulo 
	* @param string  $index 
	* @param string  $pasta 
	* @param string  $subpasta 
	* @param string  $imagem
	* @param boolean $saveDB

	* @param int $id 
	*/

	function UPH_deleteImage($modulo, $index, $pasta, $subpasta, $imagem, $id, $saveDB = true) {
		$db = \Config\Database::connect();

		if($saveDB == true) {
			$table 		= $modulo;
			$builder  = $db->table($table);

			$builder->set($index, '')->where('id', $id);
			$builder->update();
		}

		if(isset($subpasta) AND !empty($subpasta)) {
			$info      = pathinfo(PUBLICUPD . $pasta . '/' . $subpasta . '/' . $imagem);
			$pathImage = PUBLICUPD . $pasta . '/' . $subpasta . '/' . $imagem;
		}
		else {
			$info      = pathinfo(PUBLICUPD . $pasta . '/' . $imagem);
			$pathImage = PUBLICUPD . $pasta . '/' . $imagem;
		}

		if(!is_dir($pathImage) AND file_exists($pathImage)) {
			unlink($pathImage);
		}
	}

	/**
	* Remove todas as imagens, depois deleta a pasta;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param string $pasta 
	* @param int 		$id 
	* 
	* @return bool | bool 
	*/

	function UPH_deleteFolder($pasta, $id) {
		$pathFolder = PUBLICUPD . $pasta .  '/item_' . $id;

		if(is_dir($pathFolder)) {
			$files = array_diff(scandir($pathFolder), array('.', '..'));

			foreach ($files as $file) {
				(is_dir("$pathFolder/$file")) ? UPH_deleteFolder("$pathFolder/$file") : unlink("$pathFolder/$file");
			}

			return rmdir($pathFolder);
		}
		else {
			return true;
		}
	}

	/***************************************************************
	*
	*			Import
	*
	***************************************************************/

	/**
	* Faz o upload de uma única imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param string  $modulo 
	* @param string  $atual 
	* @param string  $pasta 
	* @param string  $subpasta 
	* @param string  $index 
	* @param int 	 	 $id 
	* @param int 	 	 $keyArray 
	* @param boolean $saveDB 
	* 
 	* @return strint | array
	*/

 	function UPH_importExcel($post = null, $modulo, $atual, $pasta, $subpasta, $index, $id, $keyArray = 0, $saveDB = true) {
 		$this->dataHelper = new DataHelper;
 		$db = \Config\Database::connect();

 		if(isset($_FILES[$index])) {
 			$return = array(
 				'status' => 0,
 				'msg'		 => '',
 				'file'	 => ''
 			);

 			$path = PUBLICUPD . $pasta;

 			if(!is_dir($path)) {
 				mkdir($path);
 			}

 			if(isset($subpasta) AND !empty($subpasta)) {
 				$path = PUBLICUPD . $pasta . '/' . $subpasta;

 				if(!is_dir($path)) {
 					mkdir($path);
 				}
 			}

 			if(!empty($keyArray)) {
 				$arquivo = service('request')->getFile($index . '.' . $keyArray);
 			}
 			else {
 				$arquivo = service('request')->getFile($index);
 			}

 			$mimeTypes = array(
 				'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
 			);

 			$mimeActual = $arquivo->getClientMimeType();

 			if(!in_array($mimeActual, $mimeTypes)) {
 				$return['status']  = 0;
 				$return['msg'] 		.= "O tipo de arquivo enviada é inválido. <br/>";
 			}

 			$file 			= new File($arquivo);
 			$extension 	= $file->guessExtension();

 			$config = $this->dataHelper->DH_getAjustes();

 			if($arquivo->getSizeByUnit('kb') > $config->upload_maxsize) {
 				$return['status']  = 0;
 				$return['msg'] 		.= "O tamanho do arquivo é inválido, tamanho máximo permitido: " . $config->upload_maxsize . " KB <br/>";
 			}

 			if($arquivo->isValid() AND !$arquivo->hasMoved()) {
 				$name = $arquivo->getRandomName();
 				$arquivo->move($path, $name);

 				if($saveDB == true) {
 					$table 		= $modulo;
 					$builder  = $db->table($table);

 					$builder->set($index, $name)->where('id', $id);
 					$builder->update();
 				}

 				if(file_exists($path . '/' . $name)) {
 					$fileActual = $path . '/' . $name;
 					$file 			= new File($fileActual);
 					$extension 	= $file->guessExtension();
 				}

 				$mod = $this->dataHelper->DH_getModulo($modulo);

 				if(isset($atual) AND !empty($atual)) {
 					$return['status'] = 1;
 					$return['msg'] 		= "trocou o arquivo <b>" . $atual . "</b> do módulo <b>"  . $mod->titulo . "</b> pelo <b>" . $name . '</b>. ID do cadastro: <b>' . $id . "</b>";
 					$return['file']		= $name;
 				}
 				else {
 					$return['status'] = 1;
 					$return['msg'] 		= "salvou o arquivo <b>" . $name . "</b> no módulo <b>"  . $mod->titulo . '</b>. ID do cadastro: <b>' . $id . "</b>";
 					$return['file']		= $name;
 				}
 			}
 			else {
 				$return['status']  = 0;
 				$return['msg'] 		.= "Erro ao tentar fazer o envio.";
 			}
 		}
 		else {
 			$return['status']  = 0;
 			$return['msg'] 		 = "Arquivo não encontrado.";
 		}

 		if($return['status'] == 1) {
 			ini_set('memory_limit', '-1');

 			// ---------------------------------------------------------------------------------------------------------------------------------

 			$reader 			= new \PhpOffice\PhpSpreadsheet\Reader\Xlsx();
 			$spreadSheet 	= $reader->load($fileActual);
 			$sheetNames 	= $spreadSheet->getSheetNames();
 			
 			$reader->setReadDataOnly(true);

 			// ---------------------------------------------------------------------------------------------------------------------------------
 			
 			$sheetNameMedia = "media";
 			$reader->setLoadSheetsOnly($sheetNameMedia); 
 			
 			$rowCountMedias 	= 0;
 			$workSheetMedias 	= $spreadSheet->getSheetByName($sheetNameMedia);

 			if ($workSheetMedias !== null) {
 				$tableGruposMedias = "cad_grupos_medias";
 				$builderMedia = $db->table($tableGruposMedias);
 				$builderMedia->truncate();

 				foreach ($workSheetMedias->getRowIterator() as $row) {
 					$cellIterator = $row->getCellIterator();
 					$cellIterator->setIterateOnlyExistingCells(false);

 					$value = [];
 					foreach ($cellIterator as $cell) {
 						$value[] = $cell->getValue();
 					}

 					if (array_filter($value) AND $rowCountMedias > 1) {
 						$item = array(
 							'cd_grupo'				=> $value[0],
 							'qt_contemplado'	=> $value[1],
 							'pe_lance_minimo'	=> $value[2],
 							'pe_lance_maximo'	=> $value[3],
 							'med'							=> $value[4],
 						);

 						$query = $builderMedia->insert($item);
 					}

 					$rowCountMedias++;
 				}
 			}

 			// ---------------------------------------------------------------------------------------------------------------------------------

 			$tableGruposTruncate = "cad_grupos";
 			$builderGruposTruncate = $db->table($tableGruposTruncate);
 			$builderGruposTruncate->truncate();

 			// ---------------------------------------------------------------------------------------------------------------------------------

 			$sheetNameAuto = "AUTO";
 			$reader->setLoadSheetsOnly($sheetNameAuto); 

 			$rowCountAuto 	= 0;
 			$workSheetAuto 	= $spreadSheet->getSheetByName($sheetNameAuto);

 			if ($workSheetAuto !== null) {
 				foreach ($workSheetAuto->getRowIterator() as $row) {
 					$cellIterator = $row->getCellIterator();
 					$cellIterator->setIterateOnlyExistingCells(false);

 					$value = [];
 					foreach ($cellIterator as $cell) {
 						$value[] = $cell->getValue();
 					}

 					if (array_filter($value) AND $rowCountAuto > 1) {
 						$tableGruposMedias 	= "cad_grupos_medias";
 						$builderMedia 			= $db->table($tableGruposMedias)->where('cd_grupo', $value[0]);
 						$query 							= $builderMedia->get();
 						$result 						= $query->getRow();

 						$resultContemplados = (isset($result->qt_contemplado) AND !empty($result->qt_contemplado)) ? $result->qt_contemplado : 0;
 						$resultPEMinimo 		= (isset($result->pe_lance_minimo) AND !empty($result->pe_lance_minimo)) ? $result->pe_lance_minimo : 0;
 						$resultPEMaximo 		= (isset($result->pe_lance_maximo) AND !empty($result->pe_lance_maximo)) ? $result->pe_lance_maximo : 0;

 						$value = array_filter($value);

 						$valorMaxBem 			= $this->dataHelper->DH_convertBRLtoUSD(trim($value[6]));

 						$porcTaxaAdmin		= (isset($post['postPorcTxAdmin']) AND !empty($post['postPorcTxAdmin'])) ? $post['postPorcTxAdmin'] : $value[3] ;
 						$porcFundoResv		= (isset($post['postPorcTxReserva']) AND !empty($post['postPorcTxReserva'])) ? $post['postPorcTxReserva'] : $value[4] ;
 						$porcLancEmbutido	= (isset($post['postPorcTxEmbutido']) AND !empty($post['postPorcTxEmbutido'])) ? $post['postPorcTxEmbutido'] : 30 ;
 						$porcTaxaTotal		= intval($porcTaxaAdmin) + intval($porcFundoResv);

 						$valorTaxaAdmin		= ($valorMaxBem * $porcTaxaAdmin) / 100;
 						$valorFundoResv		= ($valorMaxBem * $porcFundoResv) / 100;
 						$valorTaxaTotal		= ($valorMaxBem * $porcTaxaTotal) / 100;
 						$valorTotalPlano	= ($valorMaxBem + $valorTaxaTotal);

 						$lanceEmbutidoValor 	= ($valorTotalPlano * $porcLancEmbutido) / 100;
 						$valorCreditoLiquido 	= ($valorMaxBem - $lanceEmbutidoValor);

 						$prazoInicial		=	$value[1];
 						$parcelaInicial	=	($valorTotalPlano / $prazoInicial);

 						$itemAuto = array(
 							'titulo'							=> "Grupo: " . $value[0],
 							'id_tipo'							=> 1,
 							'grupo'								=> $value[0],
 							'prazo'								=> $value[1],
 							'vencimento'					=> $value[2],

 							'tx_admin'						=> $porcTaxaAdmin,
 							'valor_tx_admin'			=> $this->dataHelper->DH_ceilDecimal($valorTaxaAdmin),

 							'fund_res'						=> $porcFundoResv,
 							'valor_fund_res'			=> $this->dataHelper->DH_ceilDecimal($valorFundoResv),

 							'valor_min_bem'				=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[5])),
 							'valor_max_bem'				=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[6])),

 							'valor_total_plano'		=> $this->dataHelper->DH_ceilDecimal($valorTotalPlano),

 							'lanc_embutido'				=> $porcLancEmbutido,
 							'valor_lanc_embutido'	=> $this->dataHelper->DH_ceilDecimal($lanceEmbutidoValor),

 							'credito_liquido'			=> $this->dataHelper->DH_ceilDecimal($valorCreditoLiquido),

 							'menor_porc_lance'		=> $resultPEMinimo,
 							'maior_porc_lance'		=> $resultPEMaximo,

 							'contemplados'				=> $resultContemplados,

 							'parcela_inicial'			=> $parcelaInicial
 						);

 						$table 		= "cad_grupos";
 						$builder  = $db->table($table);
 						$query    = $builder->insert($itemAuto);
 					}

 					$rowCountAuto++;
 				}
 			}

 			// ---------------------------------------------------------------------------------------------------------------------------------

 			$sheetNameImovel = "IMOVEL";
 			$reader->setLoadSheetsOnly($sheetNameImovel); 

 			$rowCountImovel = 0;
 			$workSheetImovel = $spreadSheet->getSheetByName($sheetNameImovel);

 			if ($workSheetImovel !== null) {
 				foreach ($workSheetImovel->getRowIterator() as $row) {
 					$cellIterator = $row->getCellIterator();
 					$cellIterator->setIterateOnlyExistingCells(false);

 					$value = [];
 					foreach ($cellIterator as $cell) {
 						$value[] = $cell->getValue();
 					}

 					if (array_filter($value) AND $rowCountImovel > 1) {
 						$tableGruposMedias = "cad_grupos_medias";
 						$builderMedia = $db->table($tableGruposMedias)->where('cd_grupo', $value[0]);
 						$query = $builderMedia->get();
 						$result = $query->getRow();

 						$resultContemplados = (isset($result->qt_contemplado) AND !empty($result->qt_contemplado)) ? $result->qt_contemplado : 0;
 						$resultPEMinimo = (isset($result->pe_lance_minimo) AND !empty($result->pe_lance_minimo)) ? $result->pe_lance_minimo : 0;
 						$resultPEMaximo = (isset($result->pe_lance_maximo) AND !empty($result->pe_lance_maximo)) ? $result->pe_lance_maximo : 0;

 						$value = array_filter($value);

 						$valorMaxBem 			= $this->dataHelper->DH_convertBRLtoUSD(trim($value[6]));

 						$porcTaxaAdmin		= (isset($post['postPorcTxAdmin']) AND !empty($post['postPorcTxAdmin'])) ? $post['postPorcTxAdmin'] : $value[3] ;
 						$porcFundoResv		= (isset($post['postPorcTxReserva']) AND !empty($post['postPorcTxReserva'])) ? $post['postPorcTxReserva'] : $value[4] ;
 						$porcLancEmbutido	= (isset($post['postPorcTxEmbutido']) AND !empty($post['postPorcTxEmbutido'])) ? $post['postPorcTxEmbutido'] : 30 ;
 						$porcTaxaTotal		= intval($porcTaxaAdmin) + intval($porcFundoResv);

 						$valorTaxaAdmin		= ($valorMaxBem * $porcTaxaAdmin) / 100;
 						$valorFundoResv		= ($valorMaxBem * $porcFundoResv) / 100;
 						$valorTaxaTotal		= ($valorMaxBem * $porcTaxaTotal) / 100;
 						$valorTotalPlano	= ($valorMaxBem + $valorTaxaTotal);

 						$lanceEmbutidoValor 	= ($valorTotalPlano * $porcLancEmbutido) / 100;
 						$valorCreditoLiquido 	= ($valorMaxBem - $lanceEmbutidoValor);

 						$prazoInicial		=	$value[1];
 						$parcelaInicial	=	($valorTotalPlano / $prazoInicial);

 						$itemImovel = array(
 							'titulo'							=> "Grupo: " . $value[0],
 							'id_tipo'							=> 2,
 							'grupo'								=> $value[0],
 							'prazo'								=> $value[1],
 							'vencimento'					=> $value[2],

 							'tx_admin'						=> $porcTaxaAdmin,
 							'valor_tx_admin'			=> $this->dataHelper->DH_ceilDecimal($valorTaxaAdmin),

 							'fund_res'						=> $porcFundoResv,
 							'valor_fund_res'			=> $this->dataHelper->DH_ceilDecimal($valorFundoResv),

 							'valor_min_bem'				=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[5])),
 							'valor_max_bem'				=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[6])),

 							'valor_total_plano'		=> $this->dataHelper->DH_ceilDecimal($valorTotalPlano),

 							'lanc_embutido'				=> $porcLancEmbutido,
 							'valor_lanc_embutido'	=> $this->dataHelper->DH_ceilDecimal($lanceEmbutidoValor),

 							'credito_liquido'			=> $this->dataHelper->DH_ceilDecimal($valorCreditoLiquido),

 							'menor_porc_lance'		=> $resultPEMinimo,
 							'maior_porc_lance'		=> $resultPEMaximo,

 							'contemplados'				=> $resultContemplados,

 							'parcela_inicial'			=> $parcelaInicial
 						);

 						$table 		= "cad_grupos";
 						$builder  = $db->table($table);
 						$query    = $builder->insert($itemImovel);
 					}

 					$rowCountImovel++;
 				}
 			}

 			// foreach ($workSheet->getRowIterator() as $row) {
 			// 	if ($rowCount > 1) {
 			// 		// $cellIterator = $row->getCellIterator();
 			// 		// $cellIterator->setIterateOnlyExistingCells(false);

 			// 		// $value = [];
 			// 		// foreach ($cellIterator as $cell) {
 			// 		// 	$value[] = $cell->getValue();
 			// 		// }

 			// 		// echo "<pre>";
 			// 		// print_r($value);
 			// 		// echo "</pre>";
 			// 		// echo "---------------------------------------------------------------------> ";

 			// 		// $valorMaxBem 			= $this->dataHelper->DH_convertBRLtoUSD(trim($value[6]));

 			// 		// $porcTaxaAdmin		= (isset($post['postPorcTxAdmin']) AND !empty($post['postPorcTxAdmin'])) ? $post['postPorcTxAdmin'] : $value[3] ;
 			// 		// $porcFundoResv		= (isset($post['postPorcTxReserva']) AND !empty($post['postPorcTxReserva'])) ? $post['postPorcTxReserva'] : $value[4] ;
 			// 		// $porcLancEmbutido	= (isset($post['postPorcTxEmbutido']) AND !empty($post['postPorcTxEmbutido'])) ? $post['postPorcTxEmbutido'] : 30 ;
 			// 		// $porcTaxaTotal		= intval($porcTaxaAdmin) + intval($porcFundoResv);

 			// 		// $valorTaxaAdmin		= ($valorMaxBem * $porcTaxaAdmin) / 100;
 			// 		// $valorFundoResv		= ($valorMaxBem * $porcFundoResv) / 100;
 			// 		// $valorTaxaTotal		= ($valorMaxBem * $porcTaxaTotal) / 100;
 			// 		// $valorTotalPlano	= ($valorMaxBem + $valorTaxaTotal);

 			// 		// $lanceEmbutidoValor 	= ($valorTotalPlano * $porcLancEmbutido) / 100;
 			// 		// $valorCreditoLiquido 	= ($valorMaxBem - $lanceEmbutidoValor);

 			// 		// $prazoInicial		=	$value[1];
 			// 		// $parcelaInicial	=	($valorTotalPlano / $prazoInicial);

 			// 		// $data = array(
 			// 		// 	array('| Grupo encontrado.........................: ' . $value[0]),
 			// 		// 	array('| Prazo....................................: ' . $value[1]),
 			// 		// 	array('| Vencimento...............................: ' . $value[2]),

 			// 		// 	array('| Credito inicial de.......................: ' . $this->dataHelper->DH_convertUSDToBRL($valorMaxBem)),

 			// 		// 	array('| Porcentagem da Taxa administrativa.......: ' . $porcTaxaAdmin . '%'),
 			// 		// 	array('| Valor da Taxa administrativa de..........: ' . $this->dataHelper->DH_convertUSDToBRL($valorTaxaAdmin)),

 			// 		// 	array('| Porcentagem do Fundo de reserva..........: ' . $porcFundoResv . '%'),
 			// 		// 	array('| Valor do Fundo de reserva de.............: ' . $this->dataHelper->DH_convertUSDToBRL($valorFundoResv)),

 			// 		// 	array('| Porcentagem do Valor total das taxas.....: ' . $porcTaxaTotal . '%'),
 			// 		// 	array('| Valor total das taxas....................: ' . $this->dataHelper->DH_convertUSDToBRL($valorTaxaTotal)),

 			// 		// 	array('| Valor total do plano.....................: ' . $this->dataHelper->DH_convertUSDToBRL($valorTotalPlano)),

 			// 		// 	array('| Porcentagem do Valor do lance embutido...: ' . $porcLancEmbutido . '%'),
 			// 		// 	array('| Valor do lance embutido..................: ' . $this->dataHelper->DH_convertUSDToBRL($lanceEmbutidoValor)),

 			// 		// 	array('| Credito liquido apos lance embutido......: ' . $this->dataHelper->DH_convertUSDToBRL($valorCreditoLiquido)),

 			// 		// 	array('| Parcela inicial..........................: ' . $this->dataHelper->DH_convertUSDToBRL($parcelaInicial)),

 			// 		// 	array('| Contemplados (cotas).....................: ' . $value[11]),
 			// 		// );

 			// 		// echo "<pre>";
 			// 		// print_r($this->dataHelper->DH_echoPreDie($data, 75, 4));
 			// 		// echo "</pre>";

 			// 		// $item = array(
 			// 		// 	'titulo'							=> "Grupo: " . $value[0],
 			// 		// 	'grupo'								=> $value[0],
 			// 		// 	'prazo'								=> $value[1],
 			// 		// 	'vencimento'					=> $value[2],

 			// 		// 	'tx_admin'						=> $porcTaxaAdmin,
 			// 		// 	'valor_tx_admin'			=> $this->dataHelper->DH_ceilDecimal($valorTaxaAdmin),

 			// 		// 	'fund_res'						=> $porcFundoResv,
 			// 		// 	'valor_fund_res'			=> $this->dataHelper->DH_ceilDecimal($valorFundoResv),

 			// 		// 	'valor_min_bem'				=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[5])),
 			// 		// 	'valor_max_bem'				=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[6])),

 			// 		// 	'valor_total_plano'		=> $this->dataHelper->DH_ceilDecimal($valorTotalPlano),

 			// 		// 	'lanc_embutido'				=> $porcLancEmbutido,
 			// 		// 	'valor_lanc_embutido'	=> $this->dataHelper->DH_ceilDecimal($lanceEmbutidoValor),

 			// 		// 	'credito_liquido'			=> $this->dataHelper->DH_ceilDecimal($valorCreditoLiquido),

 			// 		// 	'menor_parcela_seg'		=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[7])),
 			// 		// 	'maior_parcela_seg'		=> $this->dataHelper->DH_convertBRLtoUSD(trim($value[8])),

 			// 		// 	'menor_porc_lance'		=> $value[9],
 			// 		// 	'maior_porc_lance'		=> $value[10],

 			// 		// 	'contemplados'				=> $value[11],

 			// 		// 	'parcela_inicial'			=> $parcelaInicial
 			// 		// );

 			// 		// $query = $builder->insert($item);
 			// 	}

 			// 	$rowCount++;
 			// }
 		}

 		return $return;
 	}

	/***************************************************************
	*
	*			Import
	*
	***************************************************************/

	/**
	* Faz o upload de uma única imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param string  $modulo 
	* @param string  $atual 
	* @param string  $pasta 
	* @param string  $subpasta 
	* @param string  $index 
	* @param int 	 $id 
	* @param int 	 $keyArray 
	* @param boolean $saveDB 
	* 
 	* @return strint | array
	*/

 	function UPH_uploadMultiImage($modulo, $atual, $pasta, $subpasta, $index, $subindex,  $id, $key = 0, $saveDB = true) {
 		$this->dataHelper = new DataHelper;
 		$db = \Config\Database::connect();

 		if(isset($_FILES[$index])) {

 			/*
 			* Mensagem de retorno
 			*/

 			$return = array(
 				'status' => 0,
 				'msg'		 => '',
 				'img'		 => ''
 			);

			/*
			* Valida a pasta
			*/

			$path = PUBLICUPD . $pasta;

			if(!is_dir($path)) {
				mkdir($path);
			}

			if(isset($subpasta) AND !empty($subpasta)) {
				$path = PUBLICUPD . $pasta . '/' . $subpasta;

				if(!is_dir($path)) {
					mkdir($path);
				}
			}

			/*
			* Pega o arquivo enviado
			*/

			$imagens 	= service('request')->getFiles();
			$imagem 	= $imagens[$index][$key][$subindex];

			/*
			* Valida o tipo de arquivo enviado
			*/

			$mimeTypes = array(
				'image/jpg', 
				'image/png',
				'image/vnd.adobe.photoshop',
				'image/psd',
				'image/apng',
				'image/bmp',
				'image/gif',
				'image/jpeg',
				'image/pjpeg',
				'image/tiff',
				'image/webp',
				'image/x-icon',
				'image/svg+xml',
			);

			$mimeActual = $imagem->getClientMimeType();

			if(!in_array($mimeActual, $mimeTypes)) {
				$return['status']  = 0;
				$return['msg'] 		.= "O tipo de imagem enviada é inválido. <br/>";
			}

			/*
			* Pega a extensão do arquivo
			*/

			$file 			= new File($imagem);
			$extension 	= $file->guessExtension();

			/*
			* Valida o tamanho do arquivo enviado
			*/

			$config = $this->dataHelper->DH_getAjustes();

			if($imagem->getSizeByUnit('kb') > $config->upload_maxsize) {
				$return['status']  = 0;
				$return['msg'] 		.= "O tamanho do arquivo é inválido, tamanho máximo permitido: " . $config->upload_maxsize . " KB <br/>";
			}

			if($imagem->isValid() AND !$imagem->hasMoved()) {
				$name = $imagem->getRandomName();
				$imagem->move($path, $name);

				if(isset($atual) && $atual != "") {
					$this->UPH_deleteImage($modulo, $index, $pasta, $subpasta, $atual, $id, $saveDB);
				}

				if($saveDB == true) {
					$table 		= $modulo;
					$builder  = $db->table($table);

					$builder->set($index, $name)->where('id', $id);
					$builder->update();
				}

				if(file_exists($path . '/' . $name)) {
					$fileActual = $path . '/' . $name;

					$file 			= new File($fileActual);
					$extension 	= $file->guessExtension();
				}

				$mod = $this->dataHelper->DH_getModulo($modulo);

				if(isset($atual) AND !empty($atual)) {
					$return['status'] = 1;
					$return['msg'] 		= "trocou o arquivo <b>" . $atual . "</b> do módulo <b>"  . $mod->titulo . "</b> pelo <b>" . $name . '</b>. ID do cadastro: <b>' . $id . "</b>";
					$return['img'] 		= $name;
				}
				else {
					$return['status'] = 1;
					$return['msg'] 		= "salvou o arquivo <b>" . $name . "</b> no módulo <b>"  . $mod->titulo . '</b>. ID do cadastro: <b>' . $id . "</b>";
					$return['img'] 		= $name;
				}
			}
			else {
				$return['status']  = 0;
				$return['msg'] 		.= "Erro ao tentar fazer o envio.";
			}
		}
		else {
			$return['status']  = 0;
			$return['msg'] 		 = "Arquivo não encontrado.";
		}

		return $return;
	}

	/**
	* Faz o upload de um único arquivo;
	* 
	* @version 1.0.0.0 - 2022-07-20
	* 
	* @param string  $modulo 
	* @param string  $atual 
	* @param string  $pasta 
	* @param string  $subpasta 
	* @param string  $index 
	* @param int 	 $id 
	* @param int 	 $keyArray 
	* @param boolean $saveDB 
	* 
 	* @return strint | array
	*/

	/***************************************************************
	*
	*			Import
	*
	***************************************************************/

	function UPH_uploadFile($modulo, $atual, $pasta, $subpasta, $index, $id, $keyArray = 0, $saveDB = true, $customMimeTypes = []) {
		$this->dataHelper = new DataHelper;
		$db = \Config\Database::connect();

		$return = [
			'status' => 0,
			'msg'    => '',
			'arq'    => ''
		];

		if (!isset($_FILES[$index])) {
			$return['msg'] .= "Arquivo não encontrado.";
			return $return;
		}

		$defaultMimeTypes = [
			'application/msword',
			'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
			'application/vnd.ms-excel',
			'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
			'application/vnd.ms-powerpoint',
			'application/vnd.openxmlformats-officedocument.presentationml.presentation',
			'application/pdf',
			'application/x-zip-compressed',
			'application/zip',
			'application/octet-stream',
			'text/plain',
		];

		$mimeTypes = array_merge($defaultMimeTypes, $customMimeTypes);

		$path = PUBLICUPD . basename($pasta);

		if (!is_dir($path) && !mkdir($path, 0755, true) && !is_dir($path)) {
			$return['msg'] .= "Falha ao criar o diretório principal: " . $path;
			return $return;
		}

		if (!empty($subpasta)) {
			$path = $path . '/' . basename($subpasta);
			if (!is_dir($path) && !mkdir($path, 0755, true) && !is_dir($path)) {
				$return['msg'] .= "Falha ao criar o subdiretório: " . $path;
				return $return;
			}
		}

		$arquivo = !empty($keyArray)
		? service('request')->getFile($index . '.' . $keyArray)
		: service('request')->getFile($index);

		if (!$arquivo->isValid()) {
			$return['msg'] .= "Erro no upload do arquivo: " . $arquivo->getErrorString();
			return $return;
		}

		$mimeActual = $arquivo->getClientMimeType();
		if (!in_array($mimeActual, $mimeTypes)) {
			$return['msg'] .= "Tipo de arquivo inválido. Tipo enviado: " . $mimeActual;
			return $return;
		}

		$config = $this->dataHelper->DH_getAjustes();
		$uploadMaxSizeKB = min(
			$config->upload_maxsize,
			(int)(ini_get('upload_max_filesize')) * 1024
		);

		if ($arquivo->getSizeByUnit('kb') > $uploadMaxSizeKB) {
			$return['msg'] .= "O tamanho do arquivo é inválido. Máximo permitido: " . $uploadMaxSizeKB . " KB.";
			return $return;
		}

		$name = $arquivo->getRandomName();

		if (!$arquivo->hasMoved() && $arquivo->move($path, $name)) {
			if (!empty($atual)) {
				$this->UPH_deleteFile($modulo, $index, $pasta, $subpasta, $atual, $id);
			}

			if ($saveDB) {
				$table   = $modulo;
				$builder = $db->table($table);
				$builder->set($index, $name)->where('id', $id);

				if (!$builder->update()) {
					$return['msg'] .= "Erro ao atualizar o banco de dados.";
					return $return;
				}
			}

			$mod = $this->dataHelper->DH_getModulo($modulo);
			if (!empty($atual)) {
				$return['status'] = 1;
				$return['msg']    = "Substituiu o arquivo <b>" . $atual . "</b> do módulo <b>" . $mod->titulo . "</b> por <b>" . $name . "</b>. ID: <b>" . $id . "</b>";
			} 
			else {
				$return['status'] = 1;
				$return['msg']    = "Salvou o arquivo <b>" . $name . "</b> no módulo <b>" . $mod->titulo . "</b>. ID: <b>" . $id . "</b>";
			}

			$return['arq'] = $name;
		} 
		else {
			$return['msg'] .= "Erro ao mover o arquivo.";
		}

		return $return;
	}

	/**
	* Deleta uma única imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param string  $modulo 
	* @param string  $index 
	* @param string  $pasta 
	* @param string  $subpasta 
	* @param string  $imagem
	* @param boolean $saveDB

	* @param int $id 
	*/

	function UPH_deleteFile($modulo, $index, $pasta, $subpasta, $imagem, $id, $saveDB = true) {
		$db 			= \Config\Database::connect();

		if($saveDB == true) {
			$table 		= $modulo;
			$builder  = $db->table($table);

			$builder->set($index, '')->where('id', $id);
			$builder->update();
		}

		if(isset($subpasta) AND !empty($subpasta)) {
			$pathFile = PUBLICUPD . $pasta . '/' . $subpasta . '/' . $imagem;
		}
		else {
			$pathFile = PUBLICUPD . $pasta . '/' . $imagem;
		}

		if(!is_dir($pathFile) AND file_exists($pathFile)) {
			unlink($pathFile);
		}
	}

	/**
	* Crias os favicons, com várias dimensões;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param string $atual 
	* @param string $pasta 
	* @param int 		$id 
	* 
	* @return string | array 
	*/

	function UPH_uploadFavicon($atual, $pasta, $id) {
		if(isset($_FILES['favicon']) AND $_FILES['favicon']['error'] == 0) {
			require 'Libraries/ClassPhpIco/PHP_ICO.php';

			$db 		= \Config\Database::connect();

			$imagem = service('request')->getFile('favicon');
			$path 	= PUBLICUPD . $pasta . '/';

			if(!empty($imagem) AND ($imagem->getSize() > 0 AND $imagem->getSize() < 2000000) AND $imagem->isValid() AND !$imagem->hasMoved()) {
				$name = $imagem->getRandomName();

				$imagem->move($path, $name);

				if(isset($atual) && $atual != "") {
					$imagemAtual = $path . $atual;

					if(file_exists($imagemAtual)) {
						unlink($imagemAtual);
					}
				}

				$table 		= "cad_institucional";
				$builder  = $db->table($table);

				$builder->set('favicon', $name)->where('id', $id);
				$builder->update();

				/*
				* Criandos os favicons
				*/

				$source = $path . $name;

				/*
	      * Apple
	      */

				$sizesApple = array(
					array(57, 57),
					array(60, 60),
					array(72, 72),
					array(76, 76),
					array(114, 114),
					array(120, 120),
					array(144, 144),
					array(152, 152),
					array(180, 180)
				);

				foreach ($sizesApple as $apple) {
					$destination = $path . 'favs/apple-icon-' . $apple[0] . 'x' . $apple[1] . '.png';
					$ico         = $this->UPH_resizeImage($source, $apple[0], $apple[1], $destination);
					unset($ico);
				}

				/*
	      * Android
	      */

				$sizesAndroid = array(
					array(192, 192)
				);

				foreach ($sizesAndroid as $android) {
					$destination = $path . 'favs/android-icon-' . $android[0] . 'x' . $android[1] . '.png';
					$ico         = $this->UPH_resizeImage($source, $android[0], $android[1], $destination);
					unset($ico);
				}

				/*
	      * Favicon
	      */

				$sizesFavicon = array(
					array(32, 32),
					array(96, 96),
					array(16, 16)
				);

				foreach ($sizesFavicon as $favicon) {
					$destination = $path . 'favs/favicon-' . $favicon[0] . 'x' . $favicon[1] . '.png';
					$ico         = $this->UPH_resizeImage($source, $favicon[0], $favicon[1], $destination);
					unset($ico);
				}

				/*
	      * Microsoft
	      */

				$sizesMsapp = array(
					array(144, 144)
				);

				foreach ($sizesMsapp as $msapp) {
					$destination = $path . 'favs/ms-icon-' . $msapp[0] . 'x' . $msapp[1] . '.png';
					$ico         = $this->UPH_resizeImage($source, $msapp[0], $msapp[1], $destination);
					unset($ico);
				}

				/*
				* Link
	      */

				$sizesLink = array(
					array(72, 72),
					array(96, 96),
					array(128, 128),
					array(144, 144)
				);

				foreach ($sizesLink as $link) {
					$destination = $path . 'favs/icon-' . $link[0] . 'x' . $link[1] . '.png';
					$ico         = $this->UPH_resizeImage($source, $link[0], $link[1], $destination);
					unset($ico);
				}

			} 
			else {
				return $imagem->getError();
			}

			return $name;
		}
	}

	function UPH_resizeImage(string $filename, int $width, int $height, string $savePath): bool {
		if (file_exists($filename)) {
			if (file_exists($savePath)) {
				unlink($savePath);
			}

			$image = \Config\Services::image();
			$image->withFile($filename);
			$image->resize($width, $height, false, 'height');
			$result = $image->save($savePath);

			unset($image);

			if ($result) {
				unset($result);
				return true;
			} 
			else {
				unset($result);
				return false;
			}
		} 
		else {
			return false;
		}
	}

	/**
	* Função responsável por fazer o upload das imagens da galeria;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @uses $_POST['upUrl'] 			directly
	* @uses $_POST['upCadastro'] 	directly
	* @uses $_POST['upModulo'] 		directly
	* @uses $_POST['upTabela'] 		directly
	* @uses $_POST['upPasta'] 		directly
	* @uses $_POST['upSubPasta'] 	directly
	* @uses $_POST['upIndex'] 		directly
	* 
	* @return json | string 
	*/

	function UPH_uploadImageGallery() {
		$this->constantHelper	= new ConstantHelper;
		$this->constantHelper->CNT_dataConstant();

		/*
		* $_POST
		*/

		$upUrl      = service('request')->getPost('upUrl');
		$upCadastro = service('request')->getPost('upCadastro');
		$upModulo   = service('request')->getPost('upModulo');
		$upTabela   = service('request')->getPost('upTabela');
		$upPasta    = service('request')->getPost('upPasta');
		$upSubPasta = service('request')->getPost('upSubPasta');
		$upIndex    = service('request')->getPost('upIndex');

		$hash       = "";

		if(isset($_FILES['file']) AND $_FILES['file']['error'] == 0) {

	  	/*
	  	* Valida a pasta
	    */

	  	$path = PUBLICUPD . $upPasta;

	  	if(!is_dir($path)) {
	  		mkdir($path);
	  	}

	  	if(isset($upSubPasta) AND !empty($upSubPasta)) {
	  		$path = PUBLICUPD . $upPasta . '/' . $upSubPasta;

	  		if(!is_dir($path)) {
	  			mkdir($path);
	  		}
	  	}

	    /*
	    * Faz o upload do arquivo
	    */

	    $imagem = service('request')->getFile('file');

	    if(!empty($imagem) AND $imagem->getSize() > 0 AND $imagem->isValid() AND !$imagem->hasMoved()) {
	    	$name = $imagem->getRandomName();
	    	
	    	if(!$imagem->move($path, $name)) {
	    		$return = 0;
	    		json_encode($return);
	    	}
	    	else {
	    		$str = $upCadastro . '|' . $upModulo . '|' . $upTabela . '|' . $upPasta . '|' . $upSubPasta . '|' . $upIndex . '|' . $name . '|' . $imagem->getClientName().'|';
	    		echo $str;
	    	}
	    }
	    else {
	    	$return = 0;
	    	json_encode($return);
	    }
	  }
	}

	/**
	* Galeria de Imagens, salva o registro no banco de dados;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @uses $_POST['upCadastro'] 		directly
	* @uses $_POST['upModulo'] 			directly
	* @uses $_POST['upTabela'] 			directly
	* @uses $_POST['upPasta'] 			directly
	* @uses $_POST['upSubPasta'] 		directly
	* @uses $_POST['upIndex'] 			directly
	* @uses $_POST['upImagem'] 			directly
	* @uses $_POST['upImagemNome'] 	directly
	* 
	* @return json 
	*/

	public function UPH_saveImageGallery() {
		$this->constantHelper	= new ConstantHelper;
		$this->constantHelper->CNT_dataConstant();

		$db = \Config\Database::connect();

		/*
		* $_POST
		*/

		$upCadastro 	= service('request')->getPost('upCadastro');
		$upModulo   	= service('request')->getPost('upModulo');
		$upTabela   	= service('request')->getPost('upTabela');
		$upPasta    	= service('request')->getPost('upPasta');
		$upSubPasta 	= service('request')->getPost('upSubPasta');
		$upIndex    	= service('request')->getPost('upIndex');
		$upImagem   	= service('request')->getPost('upImagem');
		$upImagemNome = service('request')->getPost('upImagemNome');

		/*
		* Dados do Cadastro
		*/

		$table 		= $upModulo;
		$builder  = $db->table($table)->where('id', $upCadastro);
		$query    = $builder->get();

		$consulta = $query->getRow();

		/*
		* Serializa a Galeria
		*/

		$arr = array(
			'titulo' => $upImagemNome, 
			'imagem' => $upImagem
		);

		if(isset($consulta) AND !empty($consulta->$upIndex)) {
			$img = unserialize($consulta->$upIndex);
			array_push($img, str_replace('"', '', $arr));
			$img = serialize($img); 
		}
		else {
			$img[] = str_replace('"', '', $arr);
			$img   = serialize($img); 
		}

		/*
		* Salva no Banco de Dados
		*/

		$table 		= $upTabela;
		$builder  = $db->table($table);

		$builder->set($upIndex, $img)->where('id', $upCadastro);
		$builder->update();

		echo json_encode($upImagem);
	}

	/**
	* Galeria de Imagens, troca o nome da imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @uses $_POST['key'] 				directly
	* @uses $_POST['upCadastro'] 	directly
	* @uses $_POST['upModulo'] 		directly
	* @uses $_POST['upTabela'] 		directly
	* @uses $_POST['upPasta'] 		directly
	* @uses $_POST['upSubPasta'] 	directly
	* @uses $_POST['upIndex'] 		directly
	* @uses $_POST['upImagem'] 		directly
	* @uses $_POST['newName'] 		directly
	* 
	* @return json 
	*/

	public function UPH_saveNewOrder() {

		$db = \Config\Database::connect();

		/*
		* $_POST
		*/

		$upKey 							= service('request')->getPost('key');
		$upKeyVariacao			= service('request')->getPost('keyVariacao');
		$upKeyModulo				= service('request')->getPost('keyModulo');
		$upCadastro 				= service('request')->getPost('upCadastro');
		$upModulo 					= service('request')->getPost('upModulo');
		$upTabela 					= service('request')->getPost('upTabela');
		$upPasta 						= service('request')->getPost('upPasta');
		$upSubPasta 				= service('request')->getPost('upSubPasta');
		$upIndex 						= service('request')->getPost('upIndex');
		$upImagem 					= service('request')->getPost('upImagem');
		$newName 						= service('request')->getPost('newName');
		$newOrder   				= service('request')->getPost('newOrder');
		$newOrderVariation  = service('request')->getPost('newOrderVariation');
		$newOrderModule  		= service('request')->getPost('newOrderModule');

		$table 			= $upTabela;
		$builder  	= $db->table($table)->where('id', $upCadastro);
		$query    	= $builder->get();

		$item 			= $query->getRow();

		if(isset($item) AND !empty($item)) {
			if($upIndex == 	"variacoes") {
				$variacao = unserialize(base64_decode($item->variacoes));
				$imagem 	= $variacao[$upKey]['imagens'][$upKeyVariacao];

				$variacao[$upKey]['imagens'][$upKeyVariacao]['ordem'] = $newOrderVariation;
				$variacao = base64_encode(serialize($variacao));

				/*
				* Salva no Banco de Dados
				*/

				$table 		= $upTabela;
				$builder  = $db->table($table);

				$builder->set($upIndex, $variacao)->where('id', $upCadastro);
				$builder->update();
			}
			elseif($upIndex == 	"modulos") {
				$modulo = unserialize(base64_decode($item->modulos));
				$imagem = $modulo[$upKey]['modulos_imagens'][$upKeyModulo];

				$modulo[$upKey]['modulos_imagens'][$upKeyModulo]['ordem'] = $newOrderModule;
				$modulo = base64_encode(serialize($modulo));

				/*
				* Salva no Banco de Dados
				*/

				$table 		= $upTabela;
				$builder  = $db->table($table);

				$builder->set($upIndex, $modulo)->where('id', $upCadastro);
				$builder->update();
			}
			else {
				$galeria = unserialize($item->$upIndex);

				foreach ($galeria as $key => $valor) {
					if($key == $upKey) {
						if(isset($newName) AND !empty($newName)) {
							$galeria[$key]['titulo'] = $newName;
						}

						if(isset($newOrder) AND !empty($newOrder)) {
							$galeria[$key]['ordem'] = $newOrder;
						}
					}
				}

				$galeria = serialize($galeria);

				/*
				* Salva no Banco de Dados
				*/

				$table 		= $upTabela;
				$builder  = $db->table($table);

				$builder->set($upIndex, $galeria)->where('id', $upCadastro);
				$builder->update();
			}

			$msg['status']   = 1;
			$msg['mensagem'] = "Nova ordem salva com sucesso.";

			echo json_encode($msg);
		}
		else {
			$msg['status']   = 0;
			$msg['mensagem'] = "Cadastro não encontrado.";

			echo json_encode($msg);
		}
	}

	/**
	* Galeria de Imagens, troca a ordem da imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @uses $_POST['key'] 				directly
	* @uses $_POST['upCadastro'] 	directly
	* @uses $_POST['upModulo'] 		directly
	* @uses $_POST['upTabela'] 		directly
	* @uses $_POST['upPasta'] 		directly
	* @uses $_POST['upSubPasta'] 	directly
	* @uses $_POST['upIndex'] 		directly
	* @uses $_POST['upImagem'] 		directly
	* @uses $_POST['newName'] 		directly
	* 
	* @return json 
	*/

	public function UPH_saveImagemName() {

		$db = \Config\Database::connect();

		/*
		* $_POST
		*/

		$upKey 			= service('request')->getPost('key');
		$upCadastro = service('request')->getPost('upCadastro');
		$upModulo 	= service('request')->getPost('upModulo');
		$upTabela 	= service('request')->getPost('upTabela');
		$upPasta 		= service('request')->getPost('upPasta');
		$upSubPasta = service('request')->getPost('upSubPasta');
		$upIndex 		= service('request')->getPost('upIndex');
		$upImagem 	= service('request')->getPost('upImagem');
		$newName 		= service('request')->getPost('newName');
		$newOrder   = service('request')->getPost('newOrder');

		$table 			= $upTabela;
		$builder  	= $db->table($table)->where('id', $upCadastro);
		$query    	= $builder->get();

		$item 			= $query->getRow();

		if(isset($item) AND !empty($item)) {
			$galeria = unserialize($item->$upIndex);

			foreach ($galeria as $key => $valor) {
				if($key == $upKey) {
					if(isset($newName) AND !empty($newName)) {
						$galeria[$key]['titulo'] = $newName;
					}

					if(isset($newOrder) AND !empty($newOrder)) {
						$galeria[$key]['ordem'] = $newOrder;
					}
				}
			}

			$galeria = serialize($galeria);

			/*
			* Salva no Banco de Dados
			*/

			$table 		= $upTabela;
			$builder  = $db->table($table);

			$builder->set($upIndex, $galeria)->where('id', $upCadastro);
			$builder->update();

			$msg['status']   = 1;
			$msg['mensagem'] = "Novo nome salvo com sucesso.";

			echo json_encode($msg);
		}
		else {
			$msg['status']   = 0;
			$msg['mensagem'] = "Cadastro não encontrado.";

			echo json_encode($msg);
		}
	}

	/**
	* Galeria de Imagens, troca a descrição da imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @uses $_POST['key'] 				directly
	* @uses $_POST['upCadastro'] 	directly
	* @uses $_POST['upModulo'] 		directly
	* @uses $_POST['upTabela'] 		directly
	* @uses $_POST['upPasta'] 		directly
	* @uses $_POST['upSubPasta'] 	directly
	* @uses $_POST['upIndex'] 		directly
	* @uses $_POST['upImagem'] 		directly
	* @uses $_POST['newName'] 		directly
	* 
	* @return json 
	*/

	public function UPH_saveImagemDescripiton() {

		$db = \Config\Database::connect();

		/*
		* $_POST
		*/

		$upKey 					= service('request')->getPost('key');
		$upCadastro 		= service('request')->getPost('upCadastro');
		$upModulo 			= service('request')->getPost('upModulo');
		$upTabela 			= service('request')->getPost('upTabela');
		$upPasta 				= service('request')->getPost('upPasta');
		$upSubPasta 		= service('request')->getPost('upSubPasta');
		$upIndex 				= service('request')->getPost('upIndex');
		$upImagem 			= service('request')->getPost('upImagem');
		$newName 				= service('request')->getPost('newName');
		$newOrder   		= service('request')->getPost('newOrder');
		$newDescription	= service('request')->getPost('newDescription');

		$table 			= $upTabela;
		$builder  	= $db->table($table)->where('id', $upCadastro);
		$query    	= $builder->get();

		$item 			= $query->getRow();

		if(isset($item) AND !empty($item)) {
			$galeria = unserialize($item->$upIndex);

			foreach ($galeria as $key => $valor) {
				if($key == $upKey) {
					if(isset($newName) AND !empty($newName)) {
						$galeria[$key]['titulo'] = $newName;
					}

					if(isset($newOrder) AND !empty($newOrder)) {
						$galeria[$key]['ordem'] = $newOrder;
					}

					if(isset($newDescription) AND !empty($newDescription)) {
						$galeria[$key]['descricao'] = $newDescription;
					}
				}
			}

			$galeria = serialize($galeria);

			/*
			* Salva no Banco de Dados
			*/

			$table 		= $upTabela;
			$builder  = $db->table($table);

			$builder->set($upIndex, $galeria)->where('id', $upCadastro);
			$builder->update();

			$msg['status']   = 1;
			$msg['mensagem'] = "Nova descrição salva com sucesso.";

			echo json_encode($msg);
		}
		else {
			$msg['status']   = 0;
			$msg['mensagem'] = "Cadastro não encontrado.";

			echo json_encode($msg);
		}
	}

	/**
	* Galeria de Imagens, deleta a imagem;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @uses $_POST['upCadastro'] 	directly
	* @uses $_POST['upModulo'] 		directly
	* @uses $_POST['upTabela'] 		directly
	* @uses $_POST['upPasta'] 		directly
	* @uses $_POST['upSubPasta'] 	directly
	* @uses $_POST['upIndex'] 		directly
	* @uses $_POST['upImagem'] 		directly
	* 
	* @return json 
	*/

	public function UPH_deleteImageGallery() {
		$this->constantHelper	= new ConstantHelper;
		$this->constantHelper->CNT_dataConstant();
		
		$db = \Config\Database::connect();

	  /*
	  * $_POST
	  */

	  $upKey 				= service('request')->getPost('upKey');
	  $upKeyImage		= service('request')->getPost('upKeyImage');
	  $upkeygaleria	= service('request')->getPost('upkeygaleria');
	  $upidcor			= service('request')->getPost('upIdCor');
	  $upkeymodulo	= service('request')->getPost('upkeymodulo');
	  $upCadastro 	= service('request')->getPost('upCadastro');
	  $upModulo   	= service('request')->getPost('upModulo');
	  $upTabela   	= service('request')->getPost('upTabela');
	  $upPasta    	= service('request')->getPost('upPasta');
	  $upSubPasta 	= service('request')->getPost('upSubPasta');
	  $upIndex    	= service('request')->getPost('upIndex');
	  $upImagem   	= service('request')->getPost('upImagem');

	  /*
	  * Dados do Cadastro
	  */

	  $table 		= $upModulo;
	  $builder  = $db->table($table)->where('id', $upCadastro);
	  $query    = $builder->get();

	  $consulta = $query->getRow();
	  
	  /*
	  * Serializa a Galeria
	  */

	  if($upIndex == "modulos") {
	  	$modulo = unserialize(base64_decode($consulta->$upIndex));
	  	$msg		= array();

	  	unset($modulo[$upidcor][$upkeymodulo]['galeria'][$upkeygaleria]);

	  	$path = PUBLICUPD . $upPasta;

	  	if(isset($upSubPasta) AND !empty($upSubPasta)) {
	  		$info      = pathinfo(PUBLICUPD . $upPasta . '/' . $upSubPasta . '/' . $upImagem);
	  		$pathImage = PUBLICUPD . $upPasta . '/' . $upSubPasta . '/' . $upImagem;
	  	}
	  	else {
	  		$info      = pathinfo(PUBLICUPD . $upPasta . '/' . $upImagem);
	  		$pathImage = PUBLICUPD . $upPasta . '/' . $upImagem;
	  	}

	  	if(file_exists($pathImage)) {
	  		if(unlink($pathImage)) {
	  			$msg['original']['status']   = 1;
	  			$msg['original']['mensagem'] = "Imagem original removida.";
	  		}
	  		else {
	  			$msg['original']['status']   = 0;
	  			$msg['original']['mensagem'] = "Imagem original não removida.";
	  		}
	  	}
	  	else {
	  		$msg['original']['status']   = 0;
	  		$msg['original']['mensagem'] = "Imagem original não encontrada.";
	  	}

	  	$modulo = base64_encode(serialize($modulo));

	  	$table 		= $upTabela;
	  	$builder  = $db->table($table);

	  	$builder->set($upIndex, $modulo)->where('id', $upCadastro);
	  	$builder->update();

	  	echo json_encode($msg);
	  }
	  else {
	  	$galeria = unserialize($consulta->$upIndex);
	  	$msg 		 = array();

	  	foreach ($galeria as $key => $valor) {
	  		if($valor['imagem'] == $upImagem) {
	  			unset($galeria[$key]);

	  			$path = PUBLICUPD . $upPasta;

	  			if(isset($upSubPasta) AND !empty($upSubPasta)) {
	  				$info      = pathinfo(PUBLICUPD . $upPasta . '/' . $upSubPasta . '/' . $upImagem);
	  				$imageWebp = $info['filename'] . '.webp';

	  				$pathImage = PUBLICUPD . $upPasta . '/' . $upSubPasta . '/' . $upImagem;
	  				$pathWebp  = PUBLICUPD . $upPasta . '/' . $upSubPasta . '/' . $imageWebp;
	  				$pathThumb = PUBLICUPD . $upPasta . '/' . $upSubPasta . '/thumb_' . $upImagem;
	  			}
	  			else {
	  				$info      = pathinfo(PUBLICUPD . $upPasta . '/' . $upImagem);
	  				$imageWebp = $info['filename'] . '.webp';

	  				$pathImage = PUBLICUPD . $upPasta . '/' . $upImagem;
	  				$pathWebp  = PUBLICUPD . $upPasta . '/' . $imageWebp;
	  				$pathThumb = PUBLICUPD . $upPasta . '/thumb_' . $upImagem;
	  			}

	  			if(file_exists($pathImage)) {
	  				if(unlink($pathImage)) {
	  					$msg['original']['status']   = 1;
	  					$msg['original']['mensagem'] = "Imagem original removida.";
	  				}
	  				else {
	  					$msg['original']['status']   = 0;
	  					$msg['original']['mensagem'] = "Imagem original não removida.";
	  				}
	  			}
	  			else {
	  				$msg['original']['status']   = 0;
	  				$msg['original']['mensagem'] = "Imagem original não encontrada.";
	  			}

	  			if(file_exists($pathWebp)) {
	  				if(unlink($pathWebp)) {
	  					$msg['webp']['status']   = 1;
	  					$msg['webp']['mensagem'] = "Imagem .webp removida.";
	  				}
	  				else {
	  					$msg['webp']['status']   = 0;
	  					$msg['webp']['mensagem'] = "Imagem .webp não removida.";
	  				}
	  			}
	  			else {
	  				$msg['webp']['status']   = 0;
	  				$msg['webp']['mensagem'] = "Imagem .webp não encontrada.";
	  			}

	  			if(file_exists($pathThumb)) {
	  				if(unlink($pathThumb)) {
	  					$msg['thumbnail']['status']   = 1;
	  					$msg['thumbnail']['mensagem'] = "Thumbnail removido.";
	  				}
	  				else {
	  					$msg['thumbnail']['status']   = 0;
	  					$msg['thumbnail']['mensagem'] = "Thumbnail não removido.";
	  				}
	  			}
	  			else {
	  				$msg['thumbnail']['status']   = 0;
	  				$msg['thumbnail']['mensagem'] = "Thumbnail não encontrado.";
	  			}
	  		}
	  	}

	  	$galeria  = serialize($galeria);

	  	$table 		= $upTabela;
	  	$builder  = $db->table($table);

	  	$builder->set($upIndex, $galeria)->where('id', $upCadastro);
	  	$builder->update();

	  	echo json_encode($msg);
	  }
	}

	/**
	* Galeria de Imagens, ordenação dos itens;
	* 
	* @version 1.0.0.0 - 2022-07-16
	* 
	* @param  array $galeria
	* 
	* @return array 
	*/

	public function UPH_orderGallery($galeria) {
		$return = [];
		foreach($galeria as $key => $valor) {
			if(array_key_exists($valor['ordem'], $return)) {
				$while = true;
				$i = 1;
				while($while) {
					if(!array_key_exists(intval($valor['ordem']) + $i, $return)) {
						$return[intval($valor['ordem']) + $i] = $valor;
						$while = false;
					}else {
						$i++;
					}
				}
			}else {
				$return[$valor['ordem']] = $valor;
			}
		}

		return $return;
	}

	/**
	* Upload de Arquivos;
	* 
	* @version 1.0.0.0 - 2022-01-08
	* 
	* @uses $_POST['pasta'] 	 directly
	* @uses $_POST['subpasta'] 	 directly
	* @uses $_POST['tabela'] 	 directly
	* @uses $_POST['item'] 		 directly
	* @uses $_POST['coluna'] 	 directly
	* @uses $_POST['atualizaDB'] directly
	* 
	* @return json 
	*/

	public function UPH_ajaxSaveDropzone() {
		$db = \Config\Database::connect();

		$pasta 			= service('request')->getPost('pasta');
		$subpasta 	= service('request')->getPost('subpasta');
		$tabela 		= service('request')->getPost('tabela');
		$item 			= service('request')->getPost('item');
		$coluna 		= service('request')->getPost('coluna');
		$atualizaDB = service('request')->getPost('atualizaDB');

		if($atualizaDB == '1') {
			$builder = $db->table($tabela);
			$builder->where('id', $item);
			$consulta = $builder->get()->getRow();

			$arraySalva = unserialize($consulta->$coluna);
		}else {
			$arraySalva = array();
		}

		if(isset($_FILES['file']) AND $_FILES['file']['error'] == 0) {

		  	/*
		  	* Valida a pasta
		    */

		  	$path = PUBLICUPD . $pasta;

		  	if(!is_dir($path)) {
		  		mkdir($path);
		  	}

		  	if(isset($subpasta) AND !empty($subpasta)) {
		  		$path = PUBLICUPD . $pasta . '/' . $subpasta;

		  		if(!is_dir($path)) {
		  			mkdir($path);
		  		}
		  	}

		    /*
		    * Faz o upload do arquivo
		    */

		    $arquivo = service('request')->getFile('file');

		    if(!empty($arquivo) AND $arquivo->getSize() > 0 AND $arquivo->isValid() AND !$arquivo->hasMoved()) {
		    	if(!$arquivo->move($path, $arquivo->getClientName())) {
		    		$return = 0;
		    		json_encode($return);
		    	}
		    	else {
		    		if($atualizaDB == '1') {
		    			$arraySalva[] = $arquivo->getClientName();

		    			$arraySalva = serialize($arraySalva);

		    			$builder = $db->table($tabela);
		    			$builder->where('id', $item);
		    			$builder->set($coluna, $arraySalva);
		    			$builder->update();
		    		}

		    		echo json_encode(array('status'=>'success', 'nome'=>$arquivo->getClientName()));
		    	}
		    }
		    else {
		    	$return = 0;
		    	json_encode($return);
		    }
		  }
		}

	/**
	* Deletar arquivos;
	* 
	* @version 1.0.0.0 - 2022-01-08
	* 
	* @uses $_POST['pasta'] 	 		directly
	* @uses $_POST['subpasta'] 		directly
	* @uses $_POST['tabela'] 	 		directly
	* @uses $_POST['item'] 			 	directly
	* @uses $_POST['coluna'] 			directly
	* @uses $_POST['atualizaDB'] 	directly
	* 
	* @return json 
	*/

	public function UPH_ajaxDeleteDropzone() {
		$db = \Config\Database::connect();

		$arquivo 		= service('request')->getPost('name');
		$pasta 			= service('request')->getPost('pasta');
		$subpasta 	= service('request')->getPost('subpasta');
		$tabela 		= service('request')->getPost('tabela');
		$item 			= service('request')->getPost('item');
		$coluna 		= service('request')->getPost('coluna');
		$atualizaDB = service('request')->getPost('atualizaDB');

		if($atualizaDB == '1') {
			$builder = $db->table($tabela);
			$builder->where('id', $item);
			$consulta = $builder->get()->getRow();

			$arraySalva = unserialize($consulta->$coluna);
		}

		/*
		* Valida a pasta
		*/

		$path = PUBLICUPD . $pasta;

		if(!is_dir($path)) {
			mkdir($path);
		}

		if(isset($subpasta) AND !empty($subpasta)) {
			$path = PUBLICUPD . $pasta . '/' . $subpasta;

			if(!is_dir($path)) {
				mkdir($path);
			}
		}

		if(file_exists($path.'/'.$arquivo)) {
			unlink($path.'/'.$arquivo);

			if($atualizaDB == '1') {
				if(!empty($arraySalva)) {
					foreach($arraySalva as $key => $valor) {
						if($valor == $arquivo) {
							unset($arraySalva[$key]);
						}
					}
					$arraySalva = serialize($arraySalva);

					$builder = $db->table($tabela);
					$builder->where('id', $item);
					$builder->set($coluna, $arraySalva);
					$builder->update();
				}
			}

			echo json_encode(array('status'=>'success', 'message'=>'O arquivo '.$arquivo.' foi removido com sucesso!'));
		}else {
			echo json_encode(array('status'=>'error', 'message'=>'Arquivo não encontrado.'));
		}
	}

	/**
	* UPH_ajaxGetDropzone;
	* 
	* @version 1.0.0.0 - 2022-01-08
	* 
	* @uses $_POST['pasta'] 	 		directly
	* @uses $_POST['subpasta'] 		directly
	* @uses $_POST['tabela'] 	 		directly
	* @uses $_POST['item'] 			 	directly
	* @uses $_POST['coluna'] 			directly
	* @uses $_POST['atualizaDB'] 	directly
	* 
	* @return json 
	*/

	public function UPH_ajaxGetDropzone() {
		$db = \Config\Database::connect();

		$pasta 		= service('request')->getPost('pasta');
		$subpasta 	= service('request')->getPost('subpasta');

		/*
		* Valida a pasta
		*/

		$path = PUBLICUPD . $pasta;

		if(!is_dir($path)) {
			mkdir($path);
		}

		if(isset($subpasta) AND !empty($subpasta)) {
			$path = PUBLICUPD . $pasta . '/' . $subpasta;

			if(!is_dir($path)) {
				mkdir($path);
			}
		}

		$arquivos = scandir($path);
		$return = [];
		unset($arquivos[array_search('.', $arquivos, true)]);
		unset($arquivos[array_search('..', $arquivos, true)]);

		if(!empty($arquivos)) {
			foreach($arquivos as $key => $value) {
				if(is_file($path.'/'.$value)) {
					$return[$key]['name'] = $value;
					if(empty($subpasta)) {
						$return[$key]['url'] = str_replace('cms/', '', site_url('public/uploads/'.$pasta.'/'.$value));
					}else {
						$return[$key]['url'] = str_replace('cms/', '', site_url('public/uploads/'.$pasta.'/'.$subpasta.'/'.$value));
					}
					$return[$key]['size'] = filesize($path.'/'.$value);
					$return[$key]['type'] = mime_content_type($path.'/'.$value);
				}
			}
		}

		echo json_encode($return);
		die();
	}
}
